/*
  localize.c

  (c) 1998-2000 (W3C) MIT, INRIA, Keio University
  See tidy.c for the copyright notice.

  You should only need to edit this file and tidy.c
  to localize HTML tidy.
*/

/*
  Deutsche bersetzung: Winfried Schch (winfried@schoech.de) v0.3 
  mit Untersttzung der Netzschule: http://www.netzschule.de/
  Anmerkung: Diese Datei kompiliert auch mit TidyGUI
             (http://perso.wanadoo.fr/ablavier/TidyGUI/)
*/

#include "platform.h"
#include "html.h"

#ifdef TIDY_WIN32
#include "../TidyItf.h"
#endif	/* TIDY_WIN32 */

/* used to point to Web Accessibility Guidelines */
#define ACCESS_URL  "http://www.w3.org/WAI/GL"

char *release_date = "4. August 2000";

static char *currentFile; /* sasdjb 01May00 for GNU Emacs error parsing */

extern uint optionerrors;

#ifdef TIDY_WIN32
tidy_out_CallbackT* tidy_out_Callback;

void Set_tidy_out_Callback(tidy_out_CallbackT* cb)
{
	tidy_out_Callback = cb;
}

void tidy_out(FILE *fp, const char* msg, ...)
{
	static char buffer[1024];
    va_list args;
    va_start(args, msg);
    vsprintf(buffer, msg, args);
    va_end(args);
	tidy_out_Callback(fp, buffer);
}
#else	/* TIDY_WIN32 */
/*
 This routine is the single point via which
 all output is written and as such is a good
 way to interface Tidy to other code when
 embedding Tidy in a GUI application.
*/
void tidy_out(FILE *fp, const char* msg, ...)
{
    va_list args;
    va_start(args, msg);
    vfprintf(fp, msg, args);
    va_end(args);
}
#endif	/* NOT(TIDY_WIN32) */

void ReadingFromStdin(void)
{
    fprintf(stderr, "Lese HTML von der Standard-Eingabe ...\n");
}

void ShowVersion(FILE *fp)
{
    tidy_out(fp, "HTML Tidy Version vom %s, bersetzung v.0.3\n"
            "Deutsche bersetzung: Winfried Schch und die Netzschule: http://www.netzschule.de/\n"
            "Besuchen Sie http://www.w3.org/People/Raggett fr weitere Informationen\n", release_date);
}

void FileError(FILE *fp, const char *file)
{
    tidy_out(fp, "Kann Datei \"%s\" nicht ffnen!\n", file);
}

static void ReportTag(Lexer *lexer, Node *tag)
{
    if (tag)
    {
        if (tag->type == StartTag)
            tidy_out(lexer->errout, "<%s>", tag->element);
        else if (tag->type == EndTag)
            tidy_out(lexer->errout, "</%s>", tag->element);
        else if (tag->type == DocTypeTag)
            tidy_out(lexer->errout, "<!DOCTYPE>");
        else if (tag->type == TextNode)
            tidy_out(lexer->errout, "plain text");
        else
            tidy_out(lexer->errout, "%s", tag->element);
    }
}

/* lexer is not defined when this is called */
void ReportUnknownOption(char *option)
{
    optionerrors++;
    fprintf(stderr, "Warnung - unbekannte Option: %s\n", option);
}

/* lexer is not defined when this is called */
void ReportBadArgument(char *option)
{
    optionerrors++;
    fprintf(stderr, "Warnung - fehlendes oder falsch gebildetes Argument fr Option: %s\n", option);
}

static void NtoS(int n, char *str)
{
    char buf[40];
    int i;

    for (i = 0;; ++i)
    {
        buf[i] = (n % 10) + '0';

        n = n /10;

        if (n == 0)
            break;
    }

    n = i;

    while (i >= 0)
    {
        str[n-i] = buf[i];
        --i;
    }

    str[n+1] = '\0';
}

static void ReportPosition(Lexer *lexer)
{
    /* Change formatting to be parsable by GNU Emacs */
    if (Emacs)
    {
        tidy_out(lexer->errout, "%s", currentFile);
        tidy_out(lexer->errout, ":%d:", lexer->lines);
        tidy_out(lexer->errout, "%d: ", lexer->columns);
    }
    else /* traditional format */
    {
        tidy_out(lexer->errout, "Zeile %d", lexer->lines);
        tidy_out(lexer->errout, " Spalte %d - ", lexer->columns);
    }
}

void ReportEncodingError(Lexer *lexer, uint code, uint c)
{
    char buf[32];

    lexer->warnings++;

    if (ShowWarnings)
    {
        ReportPosition(lexer);

        if (code == WINDOWS_CHARS)
        {
            NtoS(c, buf);
            lexer->badChars |= WINDOWS_CHARS;
            tidy_out(lexer->errout, "Warnung: ersetze illegalen Zeichencode %s", buf);
        }

        tidy_out(lexer->errout, "\n");
    }
}

void ReportEntityError(Lexer *lexer, uint code, char *entity, int c)
{
    lexer->warnings++;

    if (ShowWarnings)
    {
        ReportPosition(lexer);

        if (code == MISSING_SEMICOLON)
        {
            tidy_out(lexer->errout, "Warnung: Entity \"%s\" endet nicht mit ';'", entity);
        }
        else if (code == UNKNOWN_ENTITY)
        {
            tidy_out(lexer->errout, "Warnung: falsch geschriebenes & (sollte &amp; sein) oder unbekanntes Entity \"%s\"", entity);
        }
        else if (code == UNESCAPED_AMPERSAND)
        {
            tidy_out(lexer->errout, "Warnung: Das Zeichen & sollte als &amp; geschrieben werden");
        }

        tidy_out(lexer->errout, "\n");
    }
}

void ReportAttrError(Lexer *lexer, Node *node, char *attr, uint code)
{
    lexer->warnings++;

    /* keep quiet after 6 errors */
    if (lexer->errors > 6)
        return;

    if (ShowWarnings)
    {
        /* on end of file adjust reported position to end of input */
        if (code == UNEXPECTED_END_OF_FILE)
        {
            lexer->lines = lexer->in->curline;
            lexer->columns = lexer->in->curcol;
        }

        ReportPosition(lexer);

        if (code == UNKNOWN_ATTRIBUTE)
            tidy_out(lexer->errout, "Warnung: unbekanntes Attribut \"%s\"", attr);
        else if (code == MISSING_ATTRIBUTE)
        {
            tidy_out(lexer->errout, "Warnung: ");
            ReportTag(lexer, node);
            tidy_out(lexer->errout, " fehlt das \"%s\"-Attribut", attr);
        }
        else if (code == MISSING_ATTR_VALUE)
        {
            tidy_out(lexer->errout, "Warnung: ");
            ReportTag(lexer, node);
            tidy_out(lexer->errout, " dem Attribut \"%s\" fehlt ein Zuweisungswert", attr);
        }
        else if (code == MISSING_IMAGEMAP)
        {
            tidy_out(lexer->errout, "Warnung: ");
            ReportTag(lexer, node);
            tidy_out(lexer->errout, " es sollte eine client-seitige Imagemap verwendet werden");
            lexer->badAccess |= MISSING_IMAGE_MAP;
        }
        else if (code == BAD_ATTRIBUTE_VALUE)
        {
            tidy_out(lexer->errout, "Warnung: ");
            ReportTag(lexer, node);
            tidy_out(lexer->errout, " unbekannter Attributwert \"%s\"", attr);
        }
        else if (code == XML_ATTRIBUTE_VALUE)
        {
            tidy_out(lexer->errout, "Warnung: ");
            ReportTag(lexer, node);
            tidy_out(lexer->errout, " besitzt XML-Attribut \"%s\"", attr);
        }
        else if (code == UNEXPECTED_GT)
        {
            tidy_out(lexer->errout, "Fehler: ");
            ReportTag(lexer, node);
            tidy_out(lexer->errout, " fehlendes '>' am Ende des Tags");
            lexer->errors++;;
        }
        else if (code == UNEXPECTED_QUOTEMARK)
        {
            tidy_out(lexer->errout, "Warnung: ");
            ReportTag(lexer, node);
            tidy_out(lexer->errout, " unerwartetes oder doppeltes Anfhrungszeichen");
        }
        else if (code == REPEATED_ATTRIBUTE)
        {
            tidy_out(lexer->errout, "Warnung: ");
            ReportTag(lexer, node);
            tidy_out(lexer->errout, " wiederholtes Attribut \"%s\"", attr);
        }
        else if (code == PROPRIETARY_ATTR_VALUE)
        {
            tidy_out(lexer->errout, "Warnung: ");
            ReportTag(lexer, node);
            tidy_out(lexer->errout, " proprietrer Attributwert \"%s\"", attr);
        }
        else if (code == UNEXPECTED_END_OF_FILE)
        {
            tidy_out(lexer->errout, "Warnung: unerwartetes Dateiende beim Parsen");
        }
        else if (code == ID_NAME_MISMATCH)
        {
            tidy_out(lexer->errout, "Warnung: ");
            ReportTag(lexer, node);
            tidy_out(lexer->errout, " id- und name-Atttributwerte passen nicht zusammen");
        }

        tidy_out(lexer->errout, "\n");
    }
    else if (code == UNEXPECTED_GT)
    {
        ReportPosition(lexer);
        tidy_out(lexer->errout, "Fehler: ");
        ReportTag(lexer, node);
        tidy_out(lexer->errout, " fehlendes '>' am Ende des Tags\n");
        lexer->errors++;;
    }
}

void ReportWarning(Lexer *lexer, Node *element, Node *node, uint code)
{
    lexer->warnings++;

    /* keep quiet after 6 errors */
    if (lexer->errors > 6)
        return;

    if (ShowWarnings)
    {
        /* on end of file adjust reported position to end of input */
        if (code == UNEXPECTED_END_OF_FILE)
        {
            lexer->lines = lexer->in->curline;
            lexer->columns = lexer->in->curcol;
        }

        ReportPosition(lexer);

        if (code == MISSING_ENDTAG_FOR)
            tidy_out(lexer->errout, "Warnung: fehlendes </%s>", element->element);
        else if (code == MISSING_ENDTAG_BEFORE)
        {
            tidy_out(lexer->errout, "Warnung: fehlendes </%s> vor ", element->element);
            ReportTag(lexer, node);
        }
        else if (code == DISCARDING_UNEXPECTED)
        {
            tidy_out(lexer->errout, "Warnung: entferne unerwartetes ");
            ReportTag(lexer, node);
        }
        else if (code == NESTED_EMPHASIS)
        {
            tidy_out(lexer->errout, "Warnung: verschachtelte emphasis ");
            ReportTag(lexer, node);
        }
        else if (code == COERCE_TO_ENDTAG)
        {
            tidy_out(lexer->errout, "Warnung: <%s> soll vermutlich </%s> heien",
                node->element, node->element);
        }
        else if (code == NON_MATCHING_ENDTAG)
        {
            tidy_out(lexer->errout, "Warnung: ersetze unerwartetes unexpected ");
            ReportTag(lexer, node);
            tidy_out(lexer->errout, " durch </%s>", element->element);
        }
        else if (code == TAG_NOT_ALLOWED_IN)
        {
            tidy_out(lexer->errout, "Warnung: ");
            ReportTag(lexer, node);
            tidy_out(lexer->errout, " ist nicht erlaubt in <%s>-Elementen", element->element);
        }
        else if (code == DOCTYPE_AFTER_TAGS)
        {
            tidy_out(lexer->errout, "Warnung: <!DOCTYPE> muss ganz am Anfang stehen");
        }
        else if (code == MISSING_STARTTAG)
            tidy_out(lexer->errout, "Warnung: fehlendes <%s>", node->element);
        else if (code == UNEXPECTED_ENDTAG)
        {
            tidy_out(lexer->errout, "Warnung: unerwartetes </%s>", node->element);

            if (element)
                tidy_out(lexer->errout, " in <%s>", element->element);
        }
        else if (code == TOO_MANY_ELEMENTS)
        {
            tidy_out(lexer->errout, "Warnung: zu viele %s-Elemente", node->element);

            if (element)
                tidy_out(lexer->errout, " in <%s>", element->element);
        }
        else if (code == USING_BR_INPLACE_OF)
        {
            tidy_out(lexer->errout, "Warnung: <br> wird benutzt anstelle von ");
            ReportTag(lexer, node);
        }
        else if (code == INSERTING_TAG)
            tidy_out(lexer->errout, "Warnung: fge fehlendes <%s> ein", node->element);
        else if (code == CANT_BE_NESTED)
        {
            tidy_out(lexer->errout, "Warnung: ");
            ReportTag(lexer, node);
            tidy_out(lexer->errout, " darf nicht verschachtelt werden");
        }
        else if (code == PROPRIETARY_ELEMENT)
        {
            tidy_out(lexer->errout, "Warnung: ");
            ReportTag(lexer, node);
            tidy_out(lexer->errout, " wird vom W3C nicht gebilligt");

            if (node->tag == tag_layer)
                lexer->badLayout |= USING_LAYER;
            else if (node->tag == tag_spacer)
                lexer->badLayout |= USING_SPACER;
            else if (node->tag == tag_nobr)
                lexer->badLayout |= USING_NOBR;
        }
        else if (code == OBSOLETE_ELEMENT)
        {
            if (element->tag && (element->tag->model & CM_OBSOLETE))
                tidy_out(lexer->errout, "Warnung: ersetze das veraltete Element ");
            else
                tidy_out(lexer->errout, "Warnung: ersetze Element ");

            ReportTag(lexer, element);
            tidy_out(lexer->errout, " durch ");
            ReportTag(lexer, node);
        }
        else if (code == TRIM_EMPTY_ELEMENT)
        {
            tidy_out(lexer->errout, "Warnung: entferne leeres ");
            ReportTag(lexer, element);
        }
        else if (code == MISSING_TITLE_ELEMENT)
            tidy_out(lexer->errout, "Warnung: fge fehlendes 'title'-Element ein");
        else if (code == ILLEGAL_NESTING)
        {
            tidy_out(lexer->errout, "Warnung: ");
            ReportTag(lexer, element);
            tidy_out(lexer->errout, " sollte nicht verschachtelt werden");
        }
        else if (code == NOFRAMES_CONTENT)
        {
            tidy_out(lexer->errout, "Warnung: ");
            ReportTag(lexer, node);
            tidy_out(lexer->errout, " sollte nicht im 'noframes'-Bereich stehen");
        }
        else if (code == INCONSISTENT_VERSION)
        {
            tidy_out(lexer->errout, "Warnung: HTML-Doctype entspricht nicht dem Inhalt der Seite");
        }
        else if (code == MALFORMED_DOCTYPE)
        {
            tidy_out(lexer->errout, "Warnung: unerwartetes \"html PUBLIC\" oder \"html SYSTEM\"");
        }
        else if (code == CONTENT_AFTER_BODY)
        {
            tidy_out(lexer->errout, "Warnung: nach </body> sollte kein Inhalt mehr stehen");
        }
        else if (code == MALFORMED_COMMENT)
        {
            tidy_out(lexer->errout, "Warnung: innerhalb eines Kommentars drfen keine doppelten Bindestriche (--) stehen");
        }
        else if (code == BAD_COMMENT_CHARS)
        {
            tidy_out(lexer->errout, "Warnung: erwarte -- oder >");
        }
        else if (code == BAD_XML_COMMENT)
        {
            tidy_out(lexer->errout, "Warnung: in XML-Kommentaren darf kein -- stehen");
        }
        else if (code == BAD_CDATA_CONTENT)
        {
            tidy_out(lexer->errout, "Warnung: '<' + '/' + Buchstabe ist hier nicht erlaubt");
        }
        else if (code == INCONSISTENT_NAMESPACE)
        {
            tidy_out(lexer->errout, "Warnung: HTML-Namensraum passt nicht zum Inhalt");
        }
        else if (code == DTYPE_NOT_UPPER_CASE)
        {
            tidy_out(lexer->errout, "Warnung: SYSTEM, PUBLIC, W3C, DTD, EN mssen gro geschrieben werden");
        }
        else if (code == UNEXPECTED_END_OF_FILE)
        {
            tidy_out(lexer->errout, "Warnung: unerwartetes Ende der Datei");
            ReportTag(lexer, element);
        }

        tidy_out(lexer->errout, "\n");
    }
}

void ReportError(Lexer *lexer, Node *element, Node *node, uint code)
{
    lexer->warnings++;

    /* keep quiet after 6 errors */
    if (lexer->errors > 6)
        return;

    lexer->errors++;

    ReportPosition(lexer);

    if (code == SUSPECTED_MISSING_QUOTE)
    {
        tidy_out(lexer->errout, "Fehler: fehlendes Anfhrungszeichen fr Attributwert");
    }
    else if (code == DUPLICATE_FRAMESET)
    {
        tidy_out(lexer->errout, "Fehler: wiederholtes FRAMESET-Element");
    }
    else if (code == UNKNOWN_ELEMENT)
    {
        tidy_out(lexer->errout, "Fehler: ");
        ReportTag(lexer, node);
        tidy_out(lexer->errout, " ist nicht bekannt!");
    }
    else if (code == UNEXPECTED_ENDTAG)  /* generated by XML docs */
    {
        tidy_out(lexer->errout, "Warnung: unerwartetes </%s>", node->element);

        if (element)
            tidy_out(lexer->errout, " in <%s>", element->element);
    }

    tidy_out(lexer->errout, "\n");
}

void ErrorSummary(Lexer *lexer)
{
    /* adjust badAccess to that its null if frames are ok */
    if (lexer->badAccess & (USING_FRAMES | USING_NOFRAMES))
    {
        if (!((lexer->badAccess & USING_FRAMES) && !(lexer->badAccess & USING_NOFRAMES)))
            lexer->badAccess &= ~(USING_FRAMES | USING_NOFRAMES);
    }

    if (lexer->badChars)
    {
        if (lexer->badChars & WINDOWS_CHARS)
        {
            tidy_out(lexer->errout, "Zeichencodes fr Microsoft Windows Fonts zwischen 128 - 159\n");
            tidy_out(lexer->errout, "werden auf anderen Betriebssystemen evtl. nicht erkannt.\n");
            tidy_out(lexer->errout, "Bitte verwenden Sie stattdessen Entities, also z.B. &trade; statt\n");
            tidy_out(lexer->errout, "Windows Zeichencode 153 (Unicode 0x2122). Beachten Sie, dass\n");
            tidy_out(lexer->errout, "nur wenige Browser die neuen Entities untersttzen.\n\n");
        }
    }

    if (lexer->badForm)
    {
        tidy_out(lexer->errout, "Sie mssen mglicherweise eins von beiden oder beide <form>- und </form>-Tags\n");
        tidy_out(lexer->errout, "verschieben. HTML-Elemente sollten ordentlich geschachtelt werden -\n");
        tidy_out(lexer->errout, "Formular-Elemente sind da keine Ausnahme. Zum Beispiel sollten Sie\n");
        tidy_out(lexer->errout, "nicht das <form>-Tag in eine Tabellenzelle und das </form> in eine andere schreiben.\n");
        tidy_out(lexer->errout, "Wenn <form> vor einer Tabelle steht, darf das </form> nicht innerhalb der Tabelle\n");
        tidy_out(lexer->errout, "stehen! Beachten Sie, dass Sie nicht ein Formular in ein anderes Formular schachteln drfen!\n\n");
    }
    
    if (lexer->badAccess)
    {
        if (lexer->badAccess & MISSING_SUMMARY)
        {
            tidy_out(lexer->errout, "Das 'summary'-Attribut sollte benutzt werden, um die Struktur der\n");
            tidy_out(lexer->errout, "Tabelle zu beschreiben. Das ist sehr hilfreich fr Menschen, die\n");
            tidy_out(lexer->errout, "nicht-visuelle Browser verwenden (z.B. fr Blinde). Die 'scope'- und\n");
            tidy_out(lexer->errout, "'header'-Attribute fr Tabellenzellen sind ntzlich, um zu spezifizieren,\n");
            tidy_out(lexer->errout, "welche Tabellenkpfe zu der jeweiligen Tabellenzelle gehren. So knnen sich\n");
            tidy_out(lexer->errout, "die Benutzer von Textbrowsern einen sinnvollen Kontext fr jede Zelle\n");            
            tidy_out(lexer->errout, " erschlieen.\n\n");
        }

        if (lexer->badAccess & MISSING_IMAGE_ALT)
        {
            tidy_out(lexer->errout, "Das 'alt'-Attribut sollte benutzt werden, um eine kurze\n");
            tidy_out(lexer->errout, "Beschreibung des Bildes anzugeben; lngere Beschreibungen sollten\n");
            tidy_out(lexer->errout, "im 'longdesc'-Attribut gegeben werden, das eine URL beinhaltet, die\n");
            tidy_out(lexer->errout, "auf die Erklrung zeigt.\n");
            tidy_out(lexer->errout, "Diese Manahmen werden fr Menschen bentigt, die einen Textbrowser\n");
            tidy_out(lexer->errout, "benutzen (also z.B. Blinde, Konsolensurfer etc.).\n\n");
        }

        if (lexer->badAccess & MISSING_IMAGE_MAP)
        {
            tidy_out(lexer->errout, "Benutzen sie client-seitige Inmagemaps statt server-seitigen.\n");
            tidy_out(lexer->errout, "Letztere sind fr Menschen, die nicht-grafische Browser benutzen,\n");
            tidy_out(lexer->errout, "nicht erreichbar. Auerdem lassen sich client-seitige Imagemaps\n");
            tidy_out(lexer->errout, "leichter einrichten und sie stellen sofortiges Feedback zum\n");
            tidy_out(lexer->errout, "Benutzer der Webseite sicher.\n\n");
        }

        if (lexer->badAccess & MISSING_LINK_ALT)
        {
            tidy_out(lexer->errout, "Fr Hypertext-Links, die innerhalb einer client-seitigen Imagemap\n");
            tidy_out(lexer->errout, "benutzt werden, sollten Sie das 'alt'-Attribut setzen, um eine\n");
            tidy_out(lexer->errout, "Textbeschreibung des Links zu geben. Sonst haben es Benutzer\n");
            tidy_out(lexer->errout, "eines Textbrowsers schwer, die Navigation Ihrer Webseite zu verstehen.\n\n");            
        }

        if ((lexer->badAccess & USING_FRAMES) && !(lexer->badAccess & USING_NOFRAMES))
        {
            tidy_out(lexer->errout, "Webseiten mit Frames bereiten Blinden Menschen und\n");
            tidy_out(lexer->errout, "Leuten, die keinen frame-fhigen Browser benutzen\n");
            tidy_out(lexer->errout, "Probleme. Eine frame-basierte Seite sollte immer ein\n");
            tidy_out(lexer->errout, "Alternativ-Layout innerhalb des 'NOFRAMES'-Elements haben.\n\n");
        }

        tidy_out(lexer->errout, "Fr weitere Ratschlge, wie Sie Ihre Seiten besser erreichbar\n");
        tidy_out(lexer->errout, "machen, besuchen Sie \"%s\". Probieren Sie vielleicht auch\n", ACCESS_URL);
        tidy_out(lexer->errout, "einmal die Seite \"http://www.cast.org/bobby/\", die einen web-\n");
        tidy_out(lexer->errout, "basierten Service darstellt, der URLs auf Erreichbarkeit prft.\n\n");
    }

    if (lexer->badLayout)
    {
        if (lexer->badLayout & USING_LAYER)
        {
            tidy_out(lexer->errout, "Es werden Cascading Style Sheets (CSS) statt des proprietren\n");
            tidy_out(lexer->errout, "<LAYER>-Elements empfohlen, um Dinge auf Ihrer Webseite zu\n");
            tidy_out(lexer->errout, "positionieren. Das LAYER-Element wird nur von wenigen\n");
            tidy_out(lexer->errout, "Herstellern untersttzt.\n\n"); 
        }

        if (lexer->badLayout & USING_SPACER)
        {
            tidy_out(lexer->errout, "Bitte benutze Sie CSS, um Leerzeichen zu kontrollieren\n");
            tidy_out(lexer->errout, "(z.B- fr Einrcken, Rnder, Zeilenabstnde). Das proprietre\n");
            tidy_out(lexer->errout, "<SPACER>-Elementwird nur von wenigen Herstellern untersttzt.\n\n");
        }

        if (lexer->badLayout & USING_FONT)
        {
            tidy_out(lexer->errout, "Sie sollten CSS benutzen, um Ihre Schriftart und Eigenschaften\n");
            tidy_out(lexer->errout, "wie Schriftgre und -farbe zu spezifizieren. Das wird die Gre\n");
            tidy_out(lexer->errout, "Ihrer HTML-Dateien reduzieren und die Wartung im Vergleich zur\n");
            tidy_out(lexer->errout, "Benutzung des <FONT>-Elements vereinfachen.\n\n");
        }

        if (lexer->badLayout & USING_NOBR)
        {
            tidy_out(lexer->errout, "Sie sollten CSS benutzen, um den Zeilenumbruch zu kontrollieren.\n");
            tidy_out(lexer->errout, "Benutzen Sie \"white-space: nowrap\" statt <NOBR>...</NOBR>, wenn\n");
            tidy_out(lexer->errout, "Sie einen Zeilenumbruch verhindern wollen.\n\n");
        }

        if (lexer->badLayout & USING_BODY)
        {
            tidy_out(lexer->errout, "Sie sollten CSS benutzen, um Seitenfarbe und Linkfarben einzustellen.\n");
        }
    }
}

void UnknownOption(FILE *errout, char c)
{
    tidy_out(errout, "unbekannte Option -%c; benutzen Sie -help, um die Optionen aufzulisten.\n", c);
}

void UnknownFile(FILE *errout, char *program, char *file)
{
    tidy_out(errout, "%s: kann Datei nicht ffnen \"%s\"\n", program, file);
}

void NeedsAuthorIntervention(FILE *errout)
{
    tidy_out(errout, "Dieses Dokument hat Fehler, die zuerst beseitigt werden mssen,\n");
    tidy_out(errout, "bevor HTML-Tidy eine aufgerumte Version generieren kann.\n\n");
}

void MissingBody(FILE *errout)
{
    tidy_out(errout, "Kann slides nicht erzeugen - dem Dokument fehlt ein <BODY>-Element.\n");
}

void ReportNumberOfSlides(FILE *errout, int count)
{
    tidy_out(errout, "%d Slides gefunden\n", count);
}

void GeneralInfo(FILE *errout)
{
    tidy_out(errout, "HTML & CSS Spezifikationen finden sich auf http://www.w3.org/\n");
    tidy_out(errout, "Um mehr ber Tidy zu lesen, besuchen Sie http://www.w3.org/People/Raggett/tidy/\n");
    tidy_out(errout, "Bitte senden Sie Fehlerbenachrichtigungen an Dave Raggett c/o <html-tidy@w3.org>\n");
    tidy_out(errout, "Bitten Sie Ihre Firma, das W3C zu untersttzen - http://www.w3.org/Consortium\n");
}

void HelloMessage(FILE *errout, char *date, char *filename)
{
    currentFile = filename;  /* for use with Gnu Emacs */

    if (wstrcmp(filename, "stdin") == 0)
        tidy_out(errout, "\nTidy (Version vom %s) - Parse Konsoleneingabe (stdin)\nAbbruch mit STRG-C\n", date);
    else
        tidy_out(errout, "\nTidy (Version vom %s) parst \"%s\"\n", date, filename);
}

void ReportVersion(FILE *errout, Lexer *lexer, char *filename, Node *doctype)
{
    unsigned int i, c;
    int state = 0;
    char *vers = HTMLVersionName(lexer);

    if (doctype)
    {
        tidy_out(errout, "\n%s: Gegebener Doctype ist \"", filename);

        for (i = doctype->start; i < doctype->end; ++i)
        {
            c = (unsigned char)lexer->lexbuf[i];

            /* look for UTF-8 multibyte character */
            if (c > 0x7F)
                 i += GetUTF8((unsigned char *)lexer->lexbuf + i, &c);

            if (c == '"')
                ++state;
            else if (state == 1)
#ifdef TIDY_WIN32
				tidy_out(errout, "%c", c);
#else	/* TIDY_WIN32 */
                putc(c, errout);
#endif	/* NOT(TIDY_WIN32) */
        }

#ifdef TIDY_WIN32
				tidy_out(errout, "\"\n");
#else	/* TIDY_WIN32 */
        putc('"', errout);
#endif	/* NOT(TIDY_WIN32) */
    }

    tidy_out(errout, "\n%s: Der Dokumenten-Inhalt sieht aus wie %s\n",
                filename, (vers ? vers : "HTML proprietary"));
}

void ReportNumWarnings(FILE *errout, Lexer *lexer)
{
    if (lexer->warnings > 0)
        tidy_out(errout, "%d Warnungen/Fehler wurden gefunden!\n\n", lexer->warnings);
    else
        tidy_out(errout, "Es wurden keine Warnungen oder Fehler gefunden!\n\n");
}

void HelpText(FILE *out, char *prog)
{
#if 0  /* old style help text */
    tidy_out(out, "%s: Datei1 Datei2 ...\n", prog);
    tidy_out(out, "Werkzeug, um HTML-Seiten aufzurumen und schn auszudrucken\n");
    tidy_out(out, "Besuchen Sie http://www.w3.org/People/Raggett/tidy/\n");
    tidy_out(out, "Optionen fr Tidy vom %s\n", release_date);
    tidy_out(out, "  -config <datei>   benutzt Optionen von der Konfigurationsdatei datei\n");
    tidy_out(out, "  -indent oder -i   rckt den Inhalt von Elementen ein\n");
    tidy_out(out, "  -omit   oder -o   lsst optionale Endtags aus\n");
    tidy_out(out, "  -wrap 72          bricht Text bei Spalte 72 um (Standard ist 68)\n");
    tidy_out(out, "  -upper  oder -u   konvertiert alle Tags zu Grobuchstaben (Standard: Kleinbuchstaben)\n");
    tidy_out(out, "  -clean  oder -c   ersetzt font-, nobr- & center-Tags durch CSS-Angaben\n");
    tidy_out(out, "  -raw              lsst Zeichen > 128 unverndert in der Ausgabe\n");
    tidy_out(out, "  -ascii            benutzt ASCII fr Ausgabe, Latin-1 fr Eingabe\n");
    tidy_out(out, "  -latin1           benutzt Latin-1 fr Eingabe UND Ausgabe\n");
    tidy_out(out, "  -iso2022          benutzt ISO2022 fr Eingabe UND Ausgabe\n");
    tidy_out(out, "  -utf8             benutzt UTF-8 fr Eingabe UND Ausgabe\n");
    tidy_out(out, "  -mac              benutzt den Apple MacRoman Zeichensatz\n");
    tidy_out(out, "  -numeric oder -n  benutzt numerische statt \"sprechende\" Entities\n");
    tidy_out(out, "  -modify  oder -m  um die Originaldatei zu modifizieren\n");
    tidy_out(out, "  -errors  oder -e  nur Fehler anzeigen\n");
    tidy_out(out, "  -quiet   oder -q  unterdrckt unwichtige Ausgaben\n");
    tidy_out(out, "  -f <datei>        schreibt Fehler in benannte <datei>\n");
    tidy_out(out, "  -xml              benutzen Sie dies, wenn die Eingabe XML ist\n");
    tidy_out(out, "  -asxml            um HTML zu XML zu konvertieren\n");
    tidy_out(out, "  -slides           to burst into slides on h2 elements\n");
    tidy_out(out, "  -version oder -v  zeigt Versionsinformationen\n");
    tidy_out(out, "  -help    oder -h  zeigt Kommandozeilenoptionen\n");
    tidy_out(out, "Eingabe/Ausgabe ist standardmig stdin bzw. stdout\n");
    tidy_out(out, "Optionen mit einem Buchstaben auer -f knnen kombiniert werden\n");
    tidy_out(out, "also z.B.:  tidy -f fehler.txt -imu beispiel.html\n");
    tidy_out(out, "Fr weitere Informationen zu HTML besuchen Sie http://www.w3.org/MarkUp\n");
#endif
    tidy_out(out, "%s: Datei1 Datei2 ...\n", prog);
    tidy_out(out, "Werkzeug, um HTML-Seiten aufzurumen und schn auszudrucken\n");
    tidy_out(out, "Besuchen Sie http://www.w3.org/People/Raggett/tidy/\n");
    tidy_out(out, "Optionen fr Tidy vom %s\n", release_date);
    tidy_out(out, "\n");

    tidy_out(out, "Prozess-Anweisungen\n");
    tidy_out(out, "--------------------\n");
    tidy_out(out, "  -indent oder -i   rcke Elementinhalt ein\n");
    tidy_out(out, "  -omit   oder -o   lsst optionale Endtags aus\n");
    tidy_out(out, "  -wrap 72          bricht Text bei Spalte 72 um (Standard ist 68)\n");
    tidy_out(out, "  -upper  oder -u   konvertiert alle Tags zu Grobuchstaben (Standard: Kleinbuchstaben)\n");
    tidy_out(out, "  -clean  oder -c   ersetzt font-, nobr- & center-Tags durch CSS-Angaben\n");
    tidy_out(out, "  -numeric oder -n  benutzt numerische statt \"sprechende\" Entities\n");
    tidy_out(out, "  -errors oder -e   nur Fehler anzeigen\n");
    tidy_out(out, "  -quiet  oder -q   unterdrckt unwichtige Ausgaben\n");
    tidy_out(out, "  -xml              benutzen Sie dies, wenn die Eingabe XML ist\n");
    tidy_out(out, "  -asxml            um HTML zu XML zu konvertieren\n");
    tidy_out(out, "  -slides           to burst into slides on h2 elements\n");
    tidy_out(out, "\n");

    tidy_out(out, "Zeichensatz-Codierung\n");
    tidy_out(out, "------------------\n");
    tidy_out(out, "  -raw              lsst Zeichen > 128 unverndert in der Ausgabe\n");
    tidy_out(out, "  -ascii            benutzt ASCII fr Ausgabe, Latin-1 fr Eingabe\n");
    tidy_out(out, "  -latin1           benutzt Latin-1 fr Eingabe UND Ausgabe\n");
    tidy_out(out, "  -iso2022          benutzt ISO2022 fr Eingabe UND Ausgabe\n");
    tidy_out(out, "  -utf8             benutzt UTF-8 fr Eingabe UND Ausgabe\n");
    tidy_out(out, "  -mac              benutzt den Apple MacRoman Zeichensatz\n");
    tidy_out(out, "\n");
    tidy_out(out, "\n");

    tidy_out(out, "Dateimanipulation\n");
    tidy_out(out, "---------------\n");
    tidy_out(out, "  -config <datei>   benutzt Optionen von der Konfigurationsdatei <datei>\n");
    tidy_out(out, "  -f <datei>        schreibt Fehler in benannte <datei>\n");
    tidy_out(out, "  -modify  oder -m  um die Originaldatei zu modifizieren\n");
    tidy_out(out, "\n");

    tidy_out(out, "Verschiedenes\n");
    tidy_out(out, "------------\n");
    tidy_out(out, "  -version oder -v  zeigt Programmversion\n");
    tidy_out(out, "  -help    oder -h  listet Kommandozeilen-Optionen auf\n");
    tidy_out(out, "\n");

    tidy_out(out, "Eingabe/Ausgabe ist standardmig stdin bzw. stdout\n");
    tidy_out(out, "Optionen mit einem Buchstaben auer -f knnen kombiniert werden\n");
    tidy_out(out, "also z.B.:  tidy -f fehler.txt -imu beispiel.html\n");
    tidy_out(out, "Fr weitere Informationen zu HTML besuchen Sie http://www.w3.org/MarkUp\n");
    tidy_out(out, "\n");
}
