#include "StdAfx.h"

#include "MainDlg.h"
#include "OutputView.h"
#include "ConfigDlg.h"
#include "TidyProxy.h"
#include "Profile.h"

//-------------------------------------------------------------------
CMainDlg::CMainDlg(const char* cmdLine) :
						m_SourceFileEdit(this, 1),
						m_pOutputView(NULL), m_pConfigDlg(NULL)
{
	list<string> args = SplitCmdLine(cmdLine);
	list<string>::iterator lit = args.begin();
	while (lit != args.end()) {
		if (*lit == "-h" || *lit == "-H") {
			string msg = "Kommandozeilenoptionen:\n"
						 "  -f <Name der Quelldatei>\n"
						 "  -c <Name der Konfigurationsdatei>";
			::MessageBox(NULL, msg.c_str(), "TidyGUI", MB_OK);
		}
		else if (*lit == "-f" || *lit == "-F") {
			++lit;
			if (lit != args.end())
				m_SourceFileName = *lit;
		}
		else if (*lit == "-c" || *lit == "-C") {
			++lit;
			if (lit != args.end())
				m_CmdLineConfigFile = *lit;
		}
		++lit;
	}
}

BOOL CMainDlg::IsDialogMsg(LPMSG lpMsg)
{
	if (m_pOutputView && m_pOutputView->IsDialogMsg(lpMsg))
		return TRUE;
	if (m_pConfigDlg && m_pConfigDlg->IsDialogMsg(lpMsg))
		return TRUE;
	return IsDialogMessage(lpMsg);
}

list<string> CMainDlg::SplitCmdLine(const char* cmdLine)
{
	// Finite State Automaton (Goto considered helpful)
	// "-f 'C:\Program Files\toto.html' -c D:\tata\titi.txt"
	// is split into:
	//		-f
	//		C:\Program Files\toto.html
	//		-c
	//		D:\tata\titi.txt

	list<string> tokenList;
	unsigned int i = 0;
	string curToken;
	int eos = strlen(cmdLine);
	char c, q;

	goto state0;

state0:		// initial state - eat whitespace
	if (i == eos)
		return tokenList;
	c = cmdLine[i++];
	if (c == ' ' || c == '\t')
		goto state0;
	else if (c == '\'' || c == '"') {
		q = c;
		goto state1;
	}
	else goto state2;

state1:		// some string with whitespace between \'s
	if (i == eos) {
		if (curToken.length() > 0)
			tokenList.push_back(curToken);
		return tokenList;
	}
	c = cmdLine[i++];
	if (c == q) {
		if (curToken.length() > 0) {
			tokenList.push_back(curToken);
			curToken = "";
		}
		goto state0;
	}
	else {
		curToken.append(1, c);
		goto state1;
	}

state2:		// simple string without whitespace
	curToken.append(1, c);
	if (i == eos) {
		if (curToken.length() > 0)
			tokenList.push_back(curToken);
		return tokenList;
	}
	c = cmdLine[i++];
	if (c == ' ' || c == '\t' || c == '\'' || c == '"') {
		if (curToken.length() > 0) {
			tokenList.push_back(curToken);
			curToken = "";
		}
		if (c == '\'' || c == '"') {
			q = c;
			goto state1;
		}
		else goto state0;
	}
	else goto state2;
}

//-------------------------------------------------------------------
LRESULT CMainDlg::OnInitDialog(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	// Set HTML Tidy's icon
	HICON hicon = LoadIcon(_Module.GetModuleInstance(),
		MAKEINTRESOURCE(IDI_TIDY));
	SetIcon(hicon);

	// 'this' will handle WM_DROPFILES msgs of m_SourceFileEdit
	m_SourceFileEdit.SubclassWindow(GetDlgItem(IDC_SOURCEF_EDIT));

	m_TidyButton.Attach(GetDlgItem(IDC_TIDY_BUTTON));
	m_TidyButton.EnableWindow(FALSE);

	// Initialize message edit area
	m_MsgEdit.Attach(GetDlgItem(IDC_MSG_EDIT));

	// Initialize image list
	m_MsgImgList.Create(16, 16, ILC_COLOR4 | ILC_MASK, 2, 2);
	HICON hMsgIcon = LoadIcon(_Module.GetModuleInstance(),
		MAKEINTRESOURCE(IDI_MSG_WARNING));
	m_MsgImgList.AddIcon(hMsgIcon);
	DeleteObject(hMsgIcon);
	hMsgIcon = LoadIcon(_Module.GetModuleInstance(),
		MAKEINTRESOURCE(IDI_MSG_ERROR));
	m_MsgImgList.AddIcon(hMsgIcon);
	DeleteObject(hMsgIcon);

	// Initialize message list
	m_MsgList.Attach(GetDlgItem(IDC_MSG_LIST));
	m_MsgList.SetImageList(m_MsgImgList, LVSIL_SMALL);
	m_MsgList.InsertColumn(0, "toto", LVCFMT_LEFT, GetSystemMetrics(SM_CXSCREEN), -1);

	// Create output window (don't show it yet)
	m_pOutputView = new COutputView(&m_TidyProxy);
	m_pOutputView->Create(*this);
	m_pOutputView->ShowWindow(SW_HIDE);

	// Minimum window size
	RECT wndRect;
	GetWindowRect(&wndRect);
	m_minWndSize.cx = wndRect.right - wndRect.left;
	m_minWndSize.cy = wndRect.bottom - wndRect.top;
	GetClientRect(&wndRect);

	// Compute other values used by OnSize()
	RECT msgEditRect;
	m_MsgEdit.GetWindowRect(&msgEditRect);
	ScreenToClient(&msgEditRect);
	m_marginLeft = msgEditRect.left;
	m_marginRight = wndRect.right - msgEditRect.right;

	m_MsgEdit_TopLeft.x = msgEditRect.left;
	m_MsgEdit_TopLeft.y = msgEditRect.top;

	RECT msgListRect;
	m_MsgList.GetWindowRect(&msgListRect);
	ScreenToClient(&msgListRect);
	m_MsgWnds_Gap = msgListRect.top - msgEditRect.bottom;
	m_marginBottom = wndRect.bottom - msgListRect.bottom;

	// Command-line options consequences
	if (m_SourceFileName.length() > 0) {
		m_SourceFileEdit.SetWindowText(m_SourceFileName.c_str());
		WriteProfileString(SOURCE_FILE_REG_KEY, m_SourceFileName.c_str());
	}
	if (m_CmdLineConfigFile.length() > 0) {
		// ensure file exists
		ifstream in(m_CmdLineConfigFile.c_str());
		if (in.good()) {
			in.close();
			m_TidyProxy.LoadConfig(m_CmdLineConfigFile.c_str());
		}
	}

	// Create configuration dialog (don't show it yet)
	m_pConfigDlg = new CConfigDlg(&m_TidyProxy);
	m_pConfigDlg->Create(*this);
	m_pConfigDlg->ShowWindow(SW_HIDE);

	// Pre-load last used config file if any and if user is OK
/*	string prevConfigFileName = GetProfileString(CONFIG_FILE_REG_KEY);
	if (prevConfigFileName.length() > 0) {
		// ensure file still exists
		ifstream in(prevConfigFileName.c_str());
		if (in.good()) {
			in.close();
			m_TidyProxy.LoadConfig(prevConfigFileName.c_str());
			string msg = "Do you want to load the last used configuration file:\n";
			msg.append(prevConfigFileName);
			msg.append("?");
			int res = MessageBox(msg.c_str(), "TidyGUI", MB_YESNO | MB_DEFBUTTON1);
			if (res == IDNO)
				m_TidyProxy.ResetConfig();
		}
	}*/

	return 1;
}

//-------------------------------------------------------------------
LRESULT CMainDlg::OnQuit(WORD wNotifyCode, WORD wID, HWND hWndCtl, BOOL& bHandled)
{
	if (m_pOutputView) {
		m_pOutputView->DestroyWindow();
		delete m_pOutputView;
		m_pOutputView = NULL;
	}
	if (m_pConfigDlg) {
		m_pConfigDlg->DestroyWindow();
		delete m_pConfigDlg;
		m_pConfigDlg = NULL;
	}
	DestroyWindow();
	PostQuitMessage(0);
	return 0;
}

//-------------------------------------------------------------------
LRESULT CMainDlg::OnBrowseSourceFile(WORD wNotifyCode, WORD wID, HWND hWndCtl, BOOL& bHandled)
{
	const char* filterSpec =
		"HTML-Dateien (.htm;.html)\0*.HTM;*.HTML\0"
		"XML-Dateien (.xml;.xsl)\0*.XML;*.XSL\0"
		"Alle Dateien\0*\0"
		"\0";

	OPENFILENAME ofn;       // common dialog box structure
	char fileName[_MAX_PATH];
	string prevFileName = GetProfileString(SOURCE_FILE_REG_KEY);
	if (prevFileName.length() > 0)
		strcpy(fileName, prevFileName.c_str());
	else fileName[0] = '\0';

	// Initialize OPENFILENAME
	ZeroMemory(&ofn, sizeof(OPENFILENAME));
	ofn.lStructSize = sizeof(OPENFILENAME);
	ofn.hwndOwner = m_hWnd;
	ofn.lpstrFilter = filterSpec;
	ofn.lpstrFile = fileName;
	ofn.nMaxFile = sizeof(fileName);
	ofn.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST;

	// Display the Open dialog box
	if (GetOpenFileName(&ofn) == TRUE) {
		m_SourceFileEdit.SetWindowText(fileName);
		m_SourceFileName = fileName;
		m_TidyButton.EnableWindow(TRUE);
		WriteProfileString(SOURCE_FILE_REG_KEY, fileName);
	}

	return 0;
}

//-------------------------------------------------------------------
LRESULT CMainDlg::OnConfig(WORD wNotifyCode, WORD wID, HWND hWndCtl, BOOL& bHandled)
{
	if (m_pConfigDlg)
		m_pConfigDlg->ShowWindow(SW_SHOW);
	return 0;
}

LRESULT CMainDlg::OnTidy(WORD wNotifyCode, WORD wID, HWND hWndCtl, BOOL& bHandled)
{
	WriteProfileString(SOURCE_FILE_REG_KEY, m_SourceFileName.c_str());

	m_MsgEdit.SetWindowText("");
	m_MsgList.DeleteAllItems();

	m_TidyProxy.Reset();
	HCURSOR hPrevCursor = SetCursor(LoadCursor(NULL, IDC_WAIT));
	try {
		m_TidyProxy.DoTidy(m_SourceFileName.c_str());
	}
	catch (...) {
		MessageBox("Huch! HTML-Tidy hat eine Exception erzeugt...\n",
			"HTML-Tidy Exception", MB_OK | MB_ICONERROR);
	}
	SetCursor(hPrevCursor);

	m_pOutputView->SetText(m_TidyProxy.GetTidyOutput().c_str());

	int nItem = 0;
	string extraInfo;
	for (list<string>::const_iterator it = m_TidyProxy.GetTidyMsgs().begin();
		it != m_TidyProxy.GetTidyMsgs().end(); ++it) {
		const string& msg = *it;
		if (msg.find("Warnung:") != string::npos) {
			m_MsgList.InsertItem(nItem, it->c_str(), 0);
			++nItem;
		}
		else if (msg.find("Fehler:") != string::npos) {
			m_MsgList.InsertItem(nItem, it->c_str(), 1);
			++nItem;
		}
		else {
			extraInfo += msg;
			extraInfo += "\r\n";
		}
	}
	if (extraInfo.length() > 0)
		m_MsgEdit.SetWindowText(extraInfo.c_str());

	return 0;
}

LRESULT CMainDlg::OnShowOutput(WORD wNotifyCode, WORD wID, HWND hWndCtl, BOOL& bHandled)
{
	if (m_pOutputView)
		m_pOutputView->ShowWindow(SW_SHOW);
	return 0;
}

//-------------------------------------------------------------------
LRESULT CMainDlg::OnGetMinMaxInfo(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	MINMAXINFO* pMMI = reinterpret_cast<MINMAXINFO*>(lParam);
	pMMI->ptMinTrackSize.x = m_minWndSize.cx;
	pMMI->ptMinTrackSize.y = m_minWndSize.cy;

	return 0;
}

LRESULT CMainDlg::OnSize(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	int fwSizeType = wParam;      // resizing flag
	int nWidth = LOWORD(lParam);  // width of client area
	int nHeight = HIWORD(lParam); // height of client area

	// Compute m_MsgEdit's new size and resize it
	RECT rc;
	int w, h;
	m_MsgEdit.GetWindowRect(&rc);
	w = nWidth - m_marginLeft - m_marginRight;
	h = (nHeight - m_marginBottom - m_MsgEdit_TopLeft.y) / 2
		- m_MsgWnds_Gap / 2;
	m_MsgEdit.SetWindowPos(NULL, 0, 0, w, h,
		SWP_NOMOVE | SWP_NOZORDER);

	// Compute m_MsgList's size and move it
	int x, y;
	x = m_MsgEdit_TopLeft.x;
	y = m_MsgEdit_TopLeft.y + h + m_MsgWnds_Gap;
	m_MsgList.MoveWindow(x, y, w, h, TRUE);

	return 0;
}

//-------------------------------------------------------------------
LRESULT CMainDlg::OnSourceFileChange(WORD wNotifyCode, WORD wID, HWND hWndCtl, BOOL& bHandled)
{
	char fileName[_MAX_PATH];
	m_SourceFileEdit.GetWindowText(fileName, _MAX_PATH);
	m_SourceFileName = fileName;
	m_TidyButton.EnableWindow(m_SourceFileName.length() > 0 ? TRUE : FALSE);

	return 0;
}

LRESULT CMainDlg::OnDropSourceFile(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	HDROP hDrop = reinterpret_cast<HDROP>(wParam);
	char fileName[_MAX_PATH];
	DragQueryFile(hDrop, 0, fileName, 256);
	m_SourceFileEdit.SetWindowText(fileName);
	m_SourceFileName = fileName;
	m_TidyButton.EnableWindow(TRUE);
	WriteProfileString(SOURCE_FILE_REG_KEY, fileName);

	return 0;
}

//-------------------------------------------------------------------
LRESULT CMainDlg::OnAbout(WORD wNotifyCode, WORD wID, HWND hWndCtl, BOOL& bHandled)
{
	CAboutDlg dlg;
	dlg.DoModal();
	return 0;
}
